/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.storageengine.dataregion.flush;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.attribute.FileAttribute;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.io.FileUtils;
import org.apache.iotdb.commons.file.SystemFileFactory;
import org.apache.iotdb.commons.utils.TestOnly;
import org.apache.iotdb.db.conf.IoTDBConfig;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.tsfile.utils.FilePathUtils;
import org.apache.tsfile.utils.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CompressionRatio {
    private static final Logger LOGGER = LoggerFactory.getLogger(CompressionRatio.class);
    private static final IoTDBConfig CONFIG = IoTDBDescriptor.getInstance().getConfig();
    public static final String COMPRESSION_RATIO_DIR = "compression_ratio";
    private static final String FILE_PREFIX_BEFORE_V121 = "Ratio-";
    public static final String FILE_PREFIX = "Compress-";
    private static final String SEPARATOR = "-";
    static final String RATIO_FILE_PATH_FORMAT = "Compress-%d-%d";
    private static AtomicLong totalMemorySize = new AtomicLong(0L);
    private long totalDiskSize = 0L;
    private Map<String, Pair<Long, Long>> dataRegionRatioMap = new ConcurrentHashMap<String, Pair<Long, Long>>();
    private File directory;
    private String oldFileName = String.format("Compress-%d-%d", 0, 0);

    private CompressionRatio() {
        this.directory = SystemFileFactory.INSTANCE.getFile(FilePathUtils.regularizePath((String)CONFIG.getSystemDir()) + COMPRESSION_RATIO_DIR);
        try {
            this.restore();
        }
        catch (IOException e) {
            LOGGER.error("restore file error caused by ", (Throwable)e);
        }
    }

    public synchronized void updateRatio(long memorySize, long diskSize, String dataRegionId) throws IOException {
        File oldDataNodeFile = SystemFileFactory.INSTANCE.getFile(this.directory, this.oldFileName);
        totalMemorySize.addAndGet(memorySize);
        this.totalDiskSize += diskSize;
        if (memorySize < 0L || totalMemorySize.get() < 0L) {
            LOGGER.warn("The compression ratio is negative, current memTableSize: {}, totalMemTableSize: {}", (Object)memorySize, (Object)totalMemorySize);
        }
        File newDataNodeFile = SystemFileFactory.INSTANCE.getFile(this.directory, String.format(Locale.ENGLISH, RATIO_FILE_PATH_FORMAT, totalMemorySize.get(), this.totalDiskSize));
        this.persist(oldDataNodeFile, newDataNodeFile);
        this.oldFileName = newDataNodeFile.getName();
        Pair dataRegionCompressionRatio = this.dataRegionRatioMap.computeIfAbsent(dataRegionId, id -> new Pair((Object)0L, (Object)0L));
        File oldDataRegionFile = SystemFileFactory.INSTANCE.getFile(this.directory, String.format(Locale.ENGLISH, RATIO_FILE_PATH_FORMAT, dataRegionCompressionRatio.getLeft(), dataRegionCompressionRatio.getRight()) + "." + dataRegionId);
        dataRegionCompressionRatio.setLeft((Object)((Long)dataRegionCompressionRatio.getLeft() + memorySize));
        dataRegionCompressionRatio.setRight((Object)((Long)dataRegionCompressionRatio.getRight() + diskSize));
        File newDataRegionFile = SystemFileFactory.INSTANCE.getFile(this.directory, String.format(Locale.ENGLISH, RATIO_FILE_PATH_FORMAT, dataRegionCompressionRatio.getLeft(), dataRegionCompressionRatio.getRight()) + "." + dataRegionId);
        this.persist(oldDataRegionFile, newDataRegionFile);
    }

    public synchronized void removeDataRegionRatio(String dataRegionId) {
        Pair<Long, Long> dataRegionCompressionRatio = this.dataRegionRatioMap.remove(dataRegionId);
        if (dataRegionCompressionRatio == null) {
            return;
        }
        File oldDataRegionFile = SystemFileFactory.INSTANCE.getFile(this.directory, String.format(Locale.ENGLISH, RATIO_FILE_PATH_FORMAT, dataRegionCompressionRatio.getLeft(), dataRegionCompressionRatio.getRight()) + "." + dataRegionId);
        if (!oldDataRegionFile.delete() && oldDataRegionFile.exists()) {
            LOGGER.warn("Can't delete old data region compression file {}", (Object)oldDataRegionFile);
        }
    }

    public double getRatio() {
        return (double)totalMemorySize.get() / (double)this.totalDiskSize;
    }

    public double getRatio(String dataRegionId) {
        Pair ratioPair = this.dataRegionRatioMap.compute(dataRegionId, (dId, oldPair) -> {
            if (oldPair == null) {
                return new Pair((Object)0L, (Object)0L);
            }
            return new Pair((Object)((Long)oldPair.left), (Object)((Long)oldPair.right));
        });
        return (double)((Long)ratioPair.left).longValue() / (double)((Long)ratioPair.right).longValue();
    }

    private void persist(File oldFile, File newFile) throws IOException {
        this.checkDirectoryExist();
        if (!oldFile.exists()) {
            Files.createFile(newFile.toPath(), new FileAttribute[0]);
            LOGGER.debug("Old ratio file {} doesn't exist, force create ratio file {}", (Object)oldFile.getAbsolutePath(), (Object)newFile.getAbsolutePath());
        } else {
            FileUtils.moveFile((File)oldFile, (File)newFile);
            LOGGER.debug("Compression ratio file updated, previous: {}, current: {}", (Object)oldFile.getAbsolutePath(), (Object)newFile.getAbsolutePath());
        }
    }

    private void checkDirectoryExist() throws IOException {
        if (!this.directory.exists()) {
            FileUtils.forceMkdir((File)this.directory);
        }
    }

    private void recoverDataRegionRatio(File[] ratioFiles) {
        if (ratioFiles == null) {
            return;
        }
        HashMap<String, Integer> validFileIndex = new HashMap<String, Integer>();
        int ratioFilesLength = ratioFiles.length;
        for (int i2 = 0; i2 < ratioFilesLength; ++i2) {
            File ratioFile = ratioFiles[i2];
            String fileName = ratioFile.getName();
            String ratioPart = fileName.substring(0, fileName.lastIndexOf("."));
            String dataRegionId = fileName.substring(fileName.lastIndexOf(".") + 1);
            String[] fileNameArray = ratioPart.split(SEPARATOR);
            if (fileNameArray.length != 3) continue;
            try {
                Pair regionRatioPair = this.dataRegionRatioMap.computeIfAbsent(dataRegionId, id -> new Pair((Object)0L, (Object)0L));
                long diskSize = Long.parseLong(fileNameArray[2]);
                if (diskSize <= (Long)regionRatioPair.getRight()) continue;
                regionRatioPair.setRight((Object)diskSize);
                regionRatioPair.setLeft((Object)Long.parseLong(fileNameArray[1]));
                validFileIndex.put(dataRegionId, i2);
                continue;
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        validFileIndex.values().forEach(i -> {
            ratioFiles[i.intValue()] = null;
        });
        for (File ratioFile : ratioFiles) {
            if (ratioFile == null || ratioFile.delete()) continue;
            LOGGER.warn("Cannot delete ratio file {}", (Object)ratioFile.getAbsolutePath());
        }
    }

    private void recoverDataNodeRatio(File[] ratioFiles) throws IOException {
        if (ratioFiles != null && ratioFiles.length > 0) {
            int maxRatioIndex = 0;
            for (int i = 0; i < ratioFiles.length; ++i) {
                String[] fileNameArray = ratioFiles[i].getName().split(SEPARATOR);
                if (fileNameArray.length != 3) continue;
                try {
                    long diskSize = Long.parseLong(fileNameArray[2]);
                    if (diskSize <= this.totalDiskSize) continue;
                    totalMemorySize = new AtomicLong(Long.parseLong(fileNameArray[1]));
                    this.totalDiskSize = diskSize;
                    maxRatioIndex = i;
                    continue;
                }
                catch (NumberFormatException diskSize) {
                    // empty catch block
                }
            }
            LOGGER.debug("After restoring from compression ratio file, total memory size = {}, total disk size = {}", (Object)totalMemorySize, (Object)this.totalDiskSize);
            this.oldFileName = ratioFiles[maxRatioIndex].getName();
            CompressionRatio.deleteRedundantFilesByIndex(ratioFiles, maxRatioIndex);
        } else {
            File[] ratioFilesBeforeV121 = this.directory.listFiles((dir, name) -> name.startsWith(FILE_PREFIX_BEFORE_V121));
            if (ratioFilesBeforeV121 != null && ratioFilesBeforeV121.length > 0) {
                int maxRatioIndex = 0;
                this.totalDiskSize = 1L;
                for (int i = 0; i < ratioFilesBeforeV121.length; ++i) {
                    String[] fileNameArray = ratioFilesBeforeV121[i].getName().split(SEPARATOR);
                    if (fileNameArray.length != 3) continue;
                    try {
                        double currentCompressRatio = Double.parseDouble(fileNameArray[1]) / Double.parseDouble(fileNameArray[2]);
                        if (!(this.getRatio() < currentCompressRatio)) continue;
                        totalMemorySize = new AtomicLong((long)currentCompressRatio);
                        maxRatioIndex = i;
                        continue;
                    }
                    catch (NumberFormatException numberFormatException) {
                        // empty catch block
                    }
                }
                CompressionRatio.deleteRedundantFilesByIndex(ratioFilesBeforeV121, maxRatioIndex);
            }
        }
    }

    void restore() throws IOException {
        if (!this.directory.exists()) {
            return;
        }
        File[] dataNodeRatioFiles = this.directory.listFiles((dir, name) -> name.startsWith(FILE_PREFIX) && !name.contains("."));
        this.recoverDataNodeRatio(dataNodeRatioFiles);
        File[] dataRegionRatioFiles = this.directory.listFiles((dir, name) -> name.startsWith(FILE_PREFIX) && name.contains("."));
        this.recoverDataRegionRatio(dataRegionRatioFiles);
    }

    public static void deleteRedundantFilesByIndex(File[] files, int index) throws IOException {
        for (int i = 0; i < files.length; ++i) {
            if (i == index) continue;
            Files.delete(files[i].toPath());
        }
    }

    @TestOnly
    void reset() throws IOException {
        if (!this.directory.exists()) {
            return;
        }
        File[] ratioFiles = this.directory.listFiles((dir, name) -> name.startsWith(FILE_PREFIX));
        if (ratioFiles == null) {
            return;
        }
        for (File file : ratioFiles) {
            Files.delete(file.toPath());
        }
        totalMemorySize = new AtomicLong(0L);
        this.totalDiskSize = 0L;
    }

    public Map<String, Pair<Long, Long>> getDataRegionRatioMap() {
        return this.dataRegionRatioMap;
    }

    public static CompressionRatio getInstance() {
        return CompressionRatioHolder.INSTANCE;
    }

    private static class CompressionRatioHolder {
        private static final CompressionRatio INSTANCE = new CompressionRatio();

        private CompressionRatioHolder() {
        }
    }
}

