/*
 * Decompiled with CFR 0.152.
 */
package io.quarkus.fs.util;

import io.quarkus.fs.util.FileSystemHelper;
import io.quarkus.fs.util.FileSystemProviders;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileStore;
import java.nio.file.FileSystem;
import java.nio.file.FileSystemAlreadyExistsException;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.WatchService;
import java.nio.file.attribute.BasicFileAttributeView;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.nio.file.attribute.UserPrincipalLookupService;
import java.nio.file.spi.FileSystemProvider;
import java.time.Instant;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.stream.Stream;
import java.util.zip.ZipError;

public class ZipUtils {
    private static final String JAR_URI_PREFIX = "jar:";
    private static final Map<String, Object> DEFAULT_OWNER_ENV = new HashMap<String, Object>();
    private static final Map<String, Object> CREATE_ENV = new HashMap<String, Object>();

    public static void unzip(Path zipFile, Path targetDir) throws IOException {
        try {
            if (!Files.exists(targetDir, new LinkOption[0])) {
                Files.createDirectories(targetDir, new FileAttribute[0]);
            }
        }
        catch (FileAlreadyExistsException fae) {
            throw new IOException("Could not create directory '" + String.valueOf(targetDir) + "' as a file already exists with the same name");
        }
        try (FileSystem zipfs = ZipUtils.newFileSystem(zipFile);){
            for (Path zipRoot : zipfs.getRootDirectories()) {
                ZipUtils.copyFromZip(zipRoot, targetDir);
            }
        }
        catch (IOException | ZipError ioe) {
            throw new IOException("Could not unzip " + String.valueOf(zipFile) + " to target dir " + String.valueOf(targetDir), ioe);
        }
    }

    public static URI toZipUri(Path zipFile) throws IOException {
        URI zipUri = zipFile.toUri();
        try {
            zipUri = new URL(JAR_URI_PREFIX + zipUri.getScheme() + "://" + zipUri.getRawPath().replace("!/", "%21/") + "!/").toURI();
        }
        catch (URISyntaxException e) {
            throw new IOException("Failed to create a JAR URI for " + String.valueOf(zipFile), e);
        }
        return zipUri;
    }

    public static void copyFromZip(final Path source, final Path target) throws IOException {
        Files.walkFileTree(source, EnumSet.of(FileVisitOption.FOLLOW_LINKS), Integer.MAX_VALUE, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                block2: {
                    Path targetDir = target.resolve(source.relativize(dir).toString());
                    try {
                        Files.copy(dir, targetDir, new CopyOption[0]);
                    }
                    catch (FileAlreadyExistsException e) {
                        if (Files.isDirectory(targetDir, new LinkOption[0])) break block2;
                        throw e;
                    }
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                Files.copy(file, target.resolve(source.relativize(file).toString()), StandardCopyOption.REPLACE_EXISTING);
                return FileVisitResult.CONTINUE;
            }
        });
    }

    public static void zip(Path src, Path zipFile) throws IOException {
        block14: {
            try (FileSystem zipfs = ZipUtils.newZip(zipFile);){
                if (Files.isDirectory(src, new LinkOption[0])) {
                    try (DirectoryStream<Path> stream = Files.newDirectoryStream(src);){
                        for (Path srcPath : stream) {
                            ZipUtils.copyToZip(src, srcPath, zipfs);
                        }
                        break block14;
                    }
                }
                Files.copy(src, zipfs.getPath(src.getFileName().toString(), new String[0]), StandardCopyOption.REPLACE_EXISTING);
            }
        }
    }

    public static FileSystem newZip(Path zipFile) throws IOException {
        return ZipUtils.newZip(zipFile, Collections.emptyMap());
    }

    public static FileSystem newZip(Path zipFile, Map<String, Object> env) throws IOException {
        Map<String, Object> effectiveEnv;
        if (Files.exists(zipFile, new LinkOption[0])) {
            effectiveEnv = DEFAULT_OWNER_ENV;
        } else {
            effectiveEnv = CREATE_ENV;
            Files.createDirectories(zipFile.getParent(), new FileAttribute[0]);
        }
        if (env != null) {
            effectiveEnv = new HashMap<String, Object>(effectiveEnv);
            effectiveEnv.putAll(env);
        }
        try {
            return FileSystemProviders.ZIP_PROVIDER.newFileSystem(ZipUtils.toZipUri(zipFile), effectiveEnv);
        }
        catch (IOException ioe) {
            throw new IOException("Failed to create a new filesystem for " + String.valueOf(zipFile), ioe);
        }
    }

    private static void copyToZip(final Path srcRoot, Path srcPath, final FileSystem zipfs) throws IOException {
        Files.walkFileTree(srcPath, EnumSet.of(FileVisitOption.FOLLOW_LINKS), Integer.MAX_VALUE, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                block2: {
                    Path targetDir = zipfs.getPath(srcRoot.relativize(dir).toString(), new String[0]);
                    try {
                        Files.copy(dir, targetDir, new CopyOption[0]);
                    }
                    catch (FileAlreadyExistsException e) {
                        if (Files.isDirectory(targetDir, new LinkOption[0])) break block2;
                        throw e;
                    }
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                Files.copy(file, zipfs.getPath(srcRoot.relativize(file).toString(), new String[0]), StandardCopyOption.REPLACE_EXISTING);
                return FileVisitResult.CONTINUE;
            }
        });
    }

    public static FileSystem newFileSystem(URI uri, Map<String, Object> env) throws IOException {
        env = new HashMap<String, Object>(env);
        env.putAll(DEFAULT_OWNER_ENV);
        try {
            return FileSystemProviders.ZIP_PROVIDER.newFileSystem(uri, env);
        }
        catch (FileSystemAlreadyExistsException e) {
            throw new IOException("fs already exists " + String.valueOf(uri), e);
        }
        catch (IOException ioe) {
            throw new IOException("Failed to create a new filesystem for " + String.valueOf(uri), ioe);
        }
    }

    public static FileSystem newFileSystem(Path path) throws IOException {
        return ZipUtils.newFileSystem(path, Collections.emptyMap());
    }

    private static FileSystem newFileSystem(Path path, Map<String, Object> env) throws IOException {
        HashMap<String, Object> tmp = new HashMap<String, Object>(DEFAULT_OWNER_ENV);
        tmp.putAll(env);
        env = tmp;
        try {
            path = FileSystemHelper.ignoreFileWriteability(path);
            return FileSystemProviders.ZIP_PROVIDER.newFileSystem(path, env);
        }
        catch (IOException ioe) {
            throw new IOException("Failed to create a new filesystem for " + String.valueOf(path), ioe);
        }
    }

    @Deprecated
    public static FileSystem newFileSystem(Path path, ClassLoader classLoader) throws IOException {
        return ZipUtils.newFileSystem(path, Collections.emptyMap());
    }

    @Deprecated(forRemoval=true, since="0.11.0")
    public static OutputStream wrapForJDK8232879(OutputStream original) {
        return original;
    }

    public static void zipReproducibly(Path src, Path zipFile, Instant entryTime) throws IOException {
        block13: {
            try (FileSystem zipfs = ZipUtils.createNewReproducibleZipFileSystem(zipFile, entryTime);){
                if (Files.isDirectory(src, new LinkOption[0])) {
                    try (Stream<Path> stream = Files.walk(src, new FileVisitOption[0]);){
                        stream.sorted().forEach(srcPath -> {
                            block6: {
                                Path targetPath = zipfs.getPath(src.relativize((Path)srcPath).toString(), new String[0]);
                                try {
                                    if (Files.isDirectory(srcPath, new LinkOption[0])) {
                                        try {
                                            Files.copy(srcPath, targetPath, new CopyOption[0]);
                                            break block6;
                                        }
                                        catch (FileAlreadyExistsException e) {
                                            if (!Files.isDirectory(targetPath, new LinkOption[0])) {
                                                throw e;
                                            }
                                            break block6;
                                        }
                                    }
                                    Files.copy(srcPath, targetPath, StandardCopyOption.REPLACE_EXISTING);
                                }
                                catch (IOException e) {
                                    throw new RuntimeException(String.format("Could not copy from %s into ZIP file %s", srcPath, zipFile));
                                }
                            }
                        });
                        break block13;
                    }
                }
                Files.copy(src, zipfs.getPath(src.getFileName().toString(), new String[0]), StandardCopyOption.REPLACE_EXISTING);
            }
        }
    }

    public static FileSystem createNewReproducibleZipFileSystem(Path zipFile, Instant entryTime) throws IOException {
        return ZipUtils.createNewReproducibleZipFileSystem(zipFile, Collections.emptyMap(), entryTime);
    }

    public static FileSystem createNewReproducibleZipFileSystem(Path zipFile, Map<String, Object> env, Instant entryTime) throws IOException {
        if (Files.exists(zipFile, new LinkOption[0])) {
            throw new IllegalArgumentException("Zip file " + String.valueOf(zipFile) + " already exists");
        }
        Files.createDirectories(zipFile.getParent(), new FileAttribute[0]);
        Map<String, Object> effectiveEnv = CREATE_ENV;
        effectiveEnv.put("enablePosixFileAttributes", "true");
        if (env != null) {
            effectiveEnv = new HashMap<String, Object>(effectiveEnv);
            effectiveEnv.putAll(env);
        }
        try {
            return new ReproducibleZipFileSystem(ZipUtils.newFileSystem(ZipUtils.toZipUri(zipFile), effectiveEnv), entryTime);
        }
        catch (IOException ioe) {
            throw new IOException("Failed to create a new filesystem for " + String.valueOf(zipFile), ioe);
        }
    }

    static {
        String user = System.getProperty("user.name");
        DEFAULT_OWNER_ENV.put("defaultOwner", user);
        DEFAULT_OWNER_ENV.put("defaultGroup", user);
        CREATE_ENV.putAll(DEFAULT_OWNER_ENV);
        CREATE_ENV.put("create", "true");
    }

    private static class ReproducibleZipFileSystem
    extends FileSystem {
        private static final Set<PosixFilePermission> DIR_UNIX_MODE = PosixFilePermissions.fromString("rwxr-xr-x");
        private static final Set<PosixFilePermission> FILE_UNIX_MODE = PosixFilePermissions.fromString("rw-r--r--");
        private final FileSystem delegate;
        private final FileTime entryTime;

        public ReproducibleZipFileSystem(FileSystem delegate, Instant entryTime) {
            this.delegate = delegate;
            this.entryTime = entryTime != null ? FileTime.fromMillis(entryTime.toEpochMilli()) : null;
        }

        public int hashCode() {
            return this.delegate.hashCode();
        }

        public boolean equals(Object obj) {
            return this.delegate.equals(obj);
        }

        @Override
        public FileSystemProvider provider() {
            return this.delegate.provider();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void close() throws IOException {
            if (this.entryTime == null) {
                this.delegate.close();
                return;
            }
            try {
                for (Path dir : this.delegate.getRootDirectories()) {
                    Stream<Path> stream = Files.walk(dir, new FileVisitOption[0]);
                    try {
                        stream.filter(path -> !"/".equals(path.toString())).forEach(path -> {
                            try {
                                Files.getFileAttributeView(path, BasicFileAttributeView.class, new LinkOption[0]).setTimes(this.entryTime, this.entryTime, this.entryTime);
                            }
                            catch (IOException e) {
                                throw new RuntimeException(String.format("Could not set time attributes on %s", path), e);
                            }
                            try {
                                if (Files.isDirectory(path, new LinkOption[0])) {
                                    Files.setPosixFilePermissions(path, DIR_UNIX_MODE);
                                } else {
                                    Files.setPosixFilePermissions(path, FILE_UNIX_MODE);
                                }
                            }
                            catch (IOException e) {
                                throw new RuntimeException(String.format("Could not set file permission on %s", path), e);
                            }
                        });
                    }
                    finally {
                        if (stream == null) continue;
                        stream.close();
                    }
                }
            }
            finally {
                this.delegate.close();
            }
        }

        @Override
        public boolean isOpen() {
            return this.delegate.isOpen();
        }

        @Override
        public boolean isReadOnly() {
            return this.delegate.isReadOnly();
        }

        @Override
        public String getSeparator() {
            return this.delegate.getSeparator();
        }

        @Override
        public Iterable<Path> getRootDirectories() {
            return this.delegate.getRootDirectories();
        }

        @Override
        public Iterable<FileStore> getFileStores() {
            return this.delegate.getFileStores();
        }

        public String toString() {
            return this.delegate.toString();
        }

        @Override
        public Set<String> supportedFileAttributeViews() {
            return this.delegate.supportedFileAttributeViews();
        }

        @Override
        public Path getPath(String first, String ... more) {
            return this.delegate.getPath(first, more);
        }

        @Override
        public PathMatcher getPathMatcher(String syntaxAndPattern) {
            return this.delegate.getPathMatcher(syntaxAndPattern);
        }

        @Override
        public UserPrincipalLookupService getUserPrincipalLookupService() {
            return this.delegate.getUserPrincipalLookupService();
        }

        @Override
        public WatchService newWatchService() throws IOException {
            return this.delegate.newWatchService();
        }
    }
}

