/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.service;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.cassandra.concurrent.ScheduledExecutors;
import org.apache.cassandra.config.CassandraRelevantProperties;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.StartupChecksOptions;
import org.apache.cassandra.exceptions.StartupException;
import org.apache.cassandra.io.util.File;
import org.apache.cassandra.schema.KeyspaceMetadata;
import org.apache.cassandra.schema.SchemaKeyspace;
import org.apache.cassandra.service.StartupCheck;
import org.apache.cassandra.service.StartupChecks;
import org.apache.cassandra.utils.Clock;
import org.apache.cassandra.utils.JsonUtils;
import org.apache.cassandra.utils.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DataResurrectionCheck
implements StartupCheck {
    private static final Logger LOGGER = LoggerFactory.getLogger(DataResurrectionCheck.class);
    public static final String HEARTBEAT_FILE_CONFIG_PROPERTY = "heartbeat_file";
    public static final String EXCLUDED_KEYSPACES_CONFIG_PROPERTY = "excluded_keyspaces";
    public static final String EXCLUDED_TABLES_CONFIG_PROPERTY = "excluded_tables";
    public static final String DEFAULT_HEARTBEAT_FILE = "cassandra-heartbeat";

    static File getHeartbeatFile(Map<String, Object> config) {
        File heartbeatFile;
        String heartbeatFileConfigValue = (String)config.get(HEARTBEAT_FILE_CONFIG_PROPERTY);
        if (heartbeatFileConfigValue != null) {
            heartbeatFile = new File(heartbeatFileConfigValue);
        } else {
            String[] dataFileLocations = DatabaseDescriptor.getLocalSystemKeyspacesDataFileLocations();
            assert (dataFileLocations.length != 0);
            heartbeatFile = new File(dataFileLocations[0], DEFAULT_HEARTBEAT_FILE);
        }
        LOGGER.trace("Resolved heartbeat file for data resurrection check: " + heartbeatFile);
        return heartbeatFile;
    }

    @Override
    public StartupChecks.StartupCheckType getStartupCheckType() {
        return StartupChecks.StartupCheckType.check_data_resurrection;
    }

    @Override
    public void execute(StartupChecksOptions options) throws StartupException {
        Heartbeat heartbeat;
        if (options.isDisabled(this.getStartupCheckType())) {
            return;
        }
        Map<String, Object> config = options.getConfig(StartupChecks.StartupCheckType.check_data_resurrection);
        File heartbeatFile = DataResurrectionCheck.getHeartbeatFile(config);
        if (!heartbeatFile.exists()) {
            LOGGER.debug("Heartbeat file {} not found! Skipping heartbeat startup check.", (Object)heartbeatFile.absolutePath());
            return;
        }
        try {
            heartbeat = Heartbeat.deserializeFromJsonFile(heartbeatFile);
        }
        catch (IOException ex) {
            throw new StartupException(3, "Failed to deserialize heartbeat file " + heartbeatFile);
        }
        if (heartbeat.lastHeartbeat == null) {
            return;
        }
        long heartbeatMillis = heartbeat.lastHeartbeat.toEpochMilli();
        ArrayList<Pair<String, String>> violations = new ArrayList<Pair<String, String>>();
        Set<String> excludedKeyspaces = this.getExcludedKeyspaces(config);
        Set<Pair<String, String>> excludedTables = this.getExcludedTables(config);
        long currentTimeMillis = Clock.Global.currentTimeMillis();
        for (String keyspace : this.getKeyspaces()) {
            if (excludedKeyspaces.contains(keyspace)) continue;
            for (TableGCPeriod userTable : this.getTablesGcPeriods(keyspace)) {
                long gcGraceMillis;
                if (excludedTables.contains(Pair.create(keyspace, userTable.table)) || heartbeatMillis + (gcGraceMillis = (long)userTable.gcPeriod * 1000L) >= currentTimeMillis) continue;
                violations.add(Pair.create(keyspace, userTable.table));
            }
        }
        if (!violations.isEmpty()) {
            String invalidTables = violations.stream().map(p -> String.format("%s.%s", p.left, p.right)).collect(Collectors.joining(","));
            String exceptionMessage = String.format("There are tables for which gc_grace_seconds is older than the lastly known time Cassandra node was up based on its heartbeat %s with timestamp %s. Cassandra node will not start as it would likely introduce data consistency issues (zombies etc). Please resolve these issues manually, then remove the heartbeat and start the node again. Invalid tables: %s", heartbeatFile, heartbeat.lastHeartbeat, invalidTables);
            throw new StartupException(1, exceptionMessage);
        }
    }

    @Override
    public void postAction(StartupChecksOptions options) {
        if (options.isEnabled(StartupChecks.StartupCheckType.check_data_resurrection)) {
            Map<String, Object> config = options.getConfig(StartupChecks.StartupCheckType.check_data_resurrection);
            File heartbeatFile = DataResurrectionCheck.getHeartbeatFile(config);
            ScheduledExecutors.scheduledTasks.scheduleAtFixedRate(() -> {
                Heartbeat heartbeat = new Heartbeat(Instant.ofEpochMilli(Clock.Global.currentTimeMillis()));
                try {
                    heartbeatFile.parent().createDirectoriesIfNotExists();
                    LOGGER.trace("writing heartbeat to file " + heartbeatFile);
                    heartbeat.serializeToJsonFile(heartbeatFile);
                }
                catch (IOException ex) {
                    LOGGER.error("Unable to serialize heartbeat to " + heartbeatFile, (Throwable)ex);
                }
            }, 0L, CassandraRelevantProperties.CHECK_DATA_RESURRECTION_HEARTBEAT_PERIOD.getInt(), TimeUnit.MILLISECONDS);
        }
    }

    @VisibleForTesting
    public Set<String> getExcludedKeyspaces(Map<String, Object> config) {
        String excludedKeyspacesConfigValue = (String)config.get(EXCLUDED_KEYSPACES_CONFIG_PROPERTY);
        if (excludedKeyspacesConfigValue == null) {
            return Collections.emptySet();
        }
        return Arrays.stream(excludedKeyspacesConfigValue.trim().split(",")).map(String::trim).collect(Collectors.toSet());
    }

    @VisibleForTesting
    public Set<Pair<String, String>> getExcludedTables(Map<String, Object> config) {
        String excludedKeyspacesConfigValue = (String)config.get(EXCLUDED_TABLES_CONFIG_PROPERTY);
        if (excludedKeyspacesConfigValue == null) {
            return Collections.emptySet();
        }
        HashSet<Pair<String, String>> pairs = new HashSet<Pair<String, String>>();
        for (String keyspaceTable : excludedKeyspacesConfigValue.trim().split(",")) {
            String[] pair = keyspaceTable.trim().split("\\.");
            if (pair.length != 2) continue;
            pairs.add(Pair.create(pair[0].trim(), pair[1].trim()));
        }
        return pairs;
    }

    @VisibleForTesting
    List<String> getKeyspaces() {
        return SchemaKeyspace.fetchNonSystemKeyspaces().stream().map(keyspaceMetadata -> keyspaceMetadata.name).collect(Collectors.toList());
    }

    @VisibleForTesting
    List<TableGCPeriod> getTablesGcPeriods(String userKeyspace) {
        Optional<KeyspaceMetadata> keyspaceMetadata = SchemaKeyspace.fetchNonSystemKeyspaces().get(userKeyspace);
        if (!keyspaceMetadata.isPresent()) {
            return Collections.emptyList();
        }
        KeyspaceMetadata ksmd = keyspaceMetadata.get();
        return ksmd.tables.stream().filter(tmd -> tmd.params.gcGraceSeconds > 0).map(tmd -> new TableGCPeriod(tmd.name, tmd.params.gcGraceSeconds)).collect(Collectors.toList());
    }

    @VisibleForTesting
    static class TableGCPeriod {
        String table;
        int gcPeriod;

        TableGCPeriod(String table, int gcPeriod) {
            this.table = table;
            this.gcPeriod = gcPeriod;
        }
    }

    @JsonIgnoreProperties(ignoreUnknown=true)
    public static class Heartbeat {
        @JsonProperty(value="last_heartbeat")
        public final Instant lastHeartbeat;

        private Heartbeat() {
            this.lastHeartbeat = null;
        }

        public Heartbeat(Instant lastHeartbeat) {
            this.lastHeartbeat = lastHeartbeat;
        }

        public void serializeToJsonFile(File outputFile) throws IOException {
            JsonUtils.serializeToJsonFile(this, outputFile);
        }

        public static Heartbeat deserializeFromJsonFile(File file) throws IOException {
            return JsonUtils.deserializeFromJsonFile(Heartbeat.class, file);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Heartbeat manifest = (Heartbeat)o;
            return Objects.equals(this.lastHeartbeat, manifest.lastHeartbeat);
        }

        public int hashCode() {
            return Objects.hash(this.lastHeartbeat);
        }
    }
}

